# -*- coding: utf-8 -*-
from __future__ import division, print_function, unicode_literals

import unittest

import testsupport as ts
from pyradlib import lcompare
from pyradlib.pyrad_proc import PIPE, Error, ProcMixin


class FalsecolorTestCase(unittest.TestCase, ProcMixin):

	def _runit(self, cmd, _in):
		failmsg = None
		procs = []
		try:
			procs = self.call_many(cmd, 'test falsecolor',
					_in=_in, out=PIPE, universal_newlines=True)
			lines = procs[-1].stdout.readlines()
		except Error as e:
			failmsg = str(e)
		finally:
			for proc in procs:
				proc.wait()
		if failmsg:
			self.fail(failmsg)
		return lines

	def test_fc_hgradient(self):
		# Best we can do without a hdr file parser:
		# * We have a picture with a horizontal gradient.
		# * Run falsecolor on it with all permutations of palettes,
		#   regular/lines/bands/posterize, and -i/-ip.
		# * Convert to text (ttyimage) and extract one scanline.
		# * Compare with expected text lines as stored here.
		# We don't currently test the legend, users will notice errors there.
		hgradpic = ts.datafile('gradients', 'h_gradient.hdr')
		pals = ('def', 'eco', 'hot', 'pm3d', 'spec')
		typeargs = (
				(['-i'], {
				 'def':
				 '                 .....,,,,:::::;;;;;;;;;;:::::::'
				 '::::,,,,,,,,,,,,,,,.......................,,,,,,'
				 ',:::::;;++??%%&&**$$@@@@@@@@@@@@\n',
				 'eco':
				 '                              ...............,,,'
				 ',,,,,,,::::::::::::::::::::;;;;;;;;;;+++++++????'
				 '?%%%%%&&&&****$$$$@@@###########\n',
				 'hot':
				 '                     .........,,,,,,::::::::::::'
				 ':;;;;;;;+++++???%%%%&&&**$$@@###################'
				 '################################\n',
				 'pm3d':
				 '                 ...............................'
				 '..................,,,,,,,,,,,,,,,::::::::;;;;;;;'
				 '++++????%%%%&&***$$@@###########\n',
				 'spec':
				 '.....,,,,,,:::::;;;;;;++++++?????%%%%%%&&&&&&&&&'
				 '&&&&&%%%%%%%%%%%%%%?????????????+++++++++++++++;'
				 ';;;;;;;;;;;;;:::::::::::::::::::\n',
				 }
				),

				(['-cl', '-ip'], {
				 'def':
				 '        ...........,..,,,,,,,::::::;;::;;;;;;;;;'
				 ';::++++++++??????,,??%%%%%%%%%%..&&&&&&&&&****,,'
				 '****$$$$$$$$$%&@@@@@@@@@########\n',
				 'eco':
				 '        ...........,  ,,,,,,,::::::..::;;;;;;;;;'
				 ';,,++++++++??????::??%%%%%%%%%%;;&&&&&&&&&****??'
				 '****$$$$$$$$$*$@@@@@@@@@########\n',
				 'hot':
				 '        ..........., .,,,,,,,::::::,:::;;;;;;;;;'
				 ';;;++++++++??????%%??%%%%%%%%%%##&&&&&&&&&****##'
				 '****$$$$$$$$$##@@@@@@@@@########\n',
				 'pm3d':
				 '        ...........,..,,,,,,,::::::..::;;;;;;;;;'
				 ';..++++++++??????.,??%%%%%%%%%%,,&&&&&&&&&****;;'
				 '****$$$$$$$$$&*@@@@@@@@@########\n',
				 'spec':
				 '     ,, ...........,;;,,,,,,,::::::%%::;;;;;;;;;'
				 ';&&++++++++??????%%??%%%%%%%%%%?+&&&&&&&&&****+;'
				 '****$$$$$$$$$::@@@@@@@@@########\n',
				 }
				),

				(['-cl', '-i'], {
				 'def':
				 '                    ..             ;;           '
				 ' ::              ,,            ..             ,,'
				 '             %&                 \n',
				 'eco':
				 '                                   ..           '
				 ' ,,              ::            ;;             ??'
				 '             *$                 \n',
				 'hot':
				 '                     .             ,:           '
				 ' ;;              %%            ##             ##'
				 '             ##                 \n',
				 'pm3d':
				 '                    ..             ..           '
				 ' ..              .,            ,,             ;;'
				 '             &*                 \n',
				 'spec':
				 '     ,,             ;;             %%           '
				 ' &&              %%            ?+             +;'
				 '             ::                 \n',
				 }
				),

				(['-cb', '-ip'], {
				 'def':
				 '        ..............,,,,,,,:::::;;;::;;;;;;;;;'
				 ';:::+++++++?????,,,??%%%%%%%%%%...&&&&&&&&****,,'
				 ',***$$$$$$$$%%&&@@@@@@@@########\n',
				 'eco':
				 '        ...........   ,,,,,,,:::::...::;;;;;;;;;'
				 ';,,,+++++++?????:::??%%%%%%%%%%;;;&&&&&&&&****??'
				 '?***$$$$$$$$**$$@@@@@@@@########\n',
				 'hot':
				 '        ...........  .,,,,,,,:::::,,:::;;;;;;;;;'
				 ';;;;+++++++?????%%%??%%%%%%%%%%###&&&&&&&&****##'
				 '#***$$$$$$$$####@@@@@@@@########\n',
				 'pm3d':
				 '        ..............,,,,,,,:::::...::;;;;;;;;;'
				 ';...+++++++?????..,??%%%%%%%%%%,,:&&&&&&&&****;;'
				 '+***$$$$$$$$&&**@@@@@@@@########\n',
				 'spec':
				 '    .,, ...........;;;,,,,,,,:::::%%%::;;;;;;;;;'
				 ';&&&+++++++?????%%%??%%%%%%%%%%?++&&&&&&&&****+;'
				 ';***$$$$$$$$;:::@@@@@@@@########\n',
				 }
				),

				(['-cb', '-i'], {
				 'def':
				 '                   ...            ;;;           '
				 ' :::            ,,,            ...            ,,'
				 ',           %%&&                \n',
				 'eco':
				 '                                  ...           '
				 ' ,,,            :::            ;;;            ??'
				 '?           **$$                \n',
				 'hot':
				 '                     .            ,,:           '
				 ' ;;;            %%%            ###            ##'
				 '#           ####                \n',
				 'pm3d':
				 '                   ...            ...           '
				 ' ...            ..,            ,,:            ;;'
				 '+           &&**                \n',
				 'spec':
				 '    .,,            ;;;            %%%           '
				 ' &&&            %%%            ?++            +;'
				 ';           ;:::                \n',
				 }
				),

				(['-cp', '-i'], {
				 'def':
				 '             ...............;;;;;;;;;;;;;;;:::::'
				 '::::::::::,,,,,,,,,,,,,,,...............,,,,,,,,'
				 ',,,,,,%%%%%%%%%%%%%%%@@@@@@@@@@@\n',
				 'eco':
				 '                            ...............,,,,,'
				 ',,,,,,,,,,:::::::::::::::;;;;;;;;;;;;;;;????????'
				 '??????***************###########\n',
				 'hot':
				 '                            ,,,,,,,,,,,,,,,;;;;;'
				 ';;;;;;;;;;%%%%%%%%%%%%%%%#######################'
				 '################################\n',
				 'pm3d':
				 '             ...................................'
				 '.........................,,,,,,,,,,,,,,,;;;;;;;;'
				 ';;;;;;&&&&&&&&&&&&&&&###########\n',
				 'spec':
				 ',,,,,,,,,,,,,;;;;;;;;;;;;;;;%%%%%%%%%%%%%%%&&&&&'
				 '&&&&&&&&&&%%%%%%%%%%%%%%%???????????????;;;;;;;;'
				 ';;;;;;::::::::::::::::::::::::::\n',
				 }
				),
				)
		for ta, expd in typeargs:
			for pal in pals:
				exp = expd[pal]
				if pal == 'def':
					pa = []
				else:
					pa = ['-pal', pal]
				hfccmd = ['falsecolor', '-lw', '0'] + pa + ta + [hgradpic]
				ttycmd = ['ttyimage']
				result = self._runit((hfccmd, ttycmd), hgradpic)
				line = result[64]
				if not exp:
					print(ta)
					print('"%s"' % line)
					#print('"%s"' % exp)
					continue
				self.assertEqual(line, exp,)


# vi: set ts=4 sw=4 :
