#!/usr/bin/env python
import numpy as np

from raytraverse.scene import Scene
from raytraverse.mapper import PlanMapper, SkyMapper
from raytraverse.sky import SkyData

from raytraverse.sampler import ISamplerArea, ISamplerSuns, Sensor
from raytraverse.renderer import Rcontrib, Rtrace
from raytraverse.integrator import SensorIntegrator



def setup():
    scn = Scene("office", "office_glz.rad")
    pm = PlanMapper("plane.rad", ptres=.5)
    skyd = SkyData("geneva.epw")
    skyd.write(scene=scn)
    sm = SkyMapper(skyd.loc, sunres=20)
    return scn, pm, sm, skyd


def sample(scn, pm, sm):
    # declare rcontrib rendering object (make sure to specify -I+)
    # note this adds a sky source
    rcontrib = Rcontrib("-I+", scene=scn.scene, adpatch=200)
    # wrap in sensor
    sensor = Sensor(rcontrib)
    # run an areasampler (saves result files and returns lightplane)
    sampler = ISamplerArea(scn, sensor, stype='sky')
    skylp = sampler.run(pm)

    # repeat for -ab 1, resetting renderer first
    Rcontrib.reset()
    Rcontrib("-I+ -ab 1", scene=scn.scene, adpatch=200)
    # since rcontrib is still referenced by the sensor/sampler, we can
    # resuse with the reloaded object, no need to declare new
    # sensor or sampler
    skydlp = sampler.repeat(skylp, 'skydcomp')
    # best practice to reset one done in case called later
    Rcontrib.reset()

    # declare rtrace renderer
    # note this does not add a source
    rtrace = Rtrace("-I+ -ab 0", scene=scn.scene)
    # need to wrap a new sensor
    sensor = Sensor(rtrace)
    # sources initialized here with each sampling
    sampler = ISamplerSuns(scn, sensor)
    sunlp = sampler.run(sm, pm)
    Rtrace.reset()

    # declare an integrator for evaluation
    itg = SensorIntegrator(skylp, sunlp, skydlp,
                           ptype=("skysun", "sun", "patch"), factors=(1, 1, -1))
    return itg


def main():
    scn, pm, sm, skyd = setup()
    itg = sample(scn, pm, sm)
    # mask to march 21st
    skyd.mask = np.arange(1896, 1920)
    # evaluate zonally (for sampled points, not a re-interpolated grid)
    zlr = itg.zonal_evaluate(skyd, pm, datainfo=True)
    # save results to file so we can use from command line
    zlr.write("results_0321.npz")


if __name__ == '__main__':
    main()